/******************** (C) COPYRIGHT 2008 STMicroelectronics ********************
* File Name          : main.c
* Author             : MCD Application Team
* Version            : V2.0.1
* Date               : 06/13/2008
* Description        : Main program body
********************************************************************************
* THE PRESENT FIRMWARE WHICH IS FOR GUIDANCE ONLY AIMS AT PROVIDING CUSTOMERS
* WITH CODING INFORMATION REGARDING THEIR PRODUCTS IN ORDER FOR THEM TO SAVE TIME.
* AS A RESULT, STMICROELECTRONICS SHALL NOT BE HELD LIABLE FOR ANY DIRECT,
* INDIRECT OR CONSEQUENTIAL DAMAGES WITH RESPECT TO ANY CLAIMS ARISING FROM THE
* CONTENT OF SUCH FIRMWARE AND/OR THE USE MADE BY CUSTOMERS OF THE CODING
* INFORMATION CONTAINED HEREIN IN CONNECTION WITH THEIR PRODUCTS.
*******************************************************************************/

/* Includes ------------------------------------------------------------------*/
#include "stm32f10x_lib.h"
#include "platform_config.h"

/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
GPIO_InitTypeDef GPIO_InitStructure;
ErrorStatus HSEStartUpStatus;
static vu32 TimingDelay;

/* Private function prototypes -----------------------------------------------*/
void RCC_Configuration(void);
void NVIC_Configuration(void);
void lcd_init(void);					// Initial LCD
void lcd_out_data4(unsigned char);		// Strobe 4-Bit Data to LCD
void lcd_write_byte(unsigned char);		// Write 1 Byte Data to LCD
void lcd_write_control(unsigned char); 	// Write Instruction
void lcd_write_ascii(unsigned char); 	// Write LCD Display(ASCII)
void goto_cursor(unsigned char);		// Set Position Cursor LCD
void lcd_print(unsigned char*);			// Print Display to LCD
char busy_lcd(void);					// Read Busy LCD Status
void enable_lcd(void);	 				// Enable Pulse
void DelayuS(vu32 nCount);	 			// 1uS Delay
void DelaymS(vu32 nTime);				// 1mS Delay
void TimingDelay_Decrement(void);

/* Private functions ---------------------------------------------------------*/

/*******************************************************************************
* Function Name  : main
* Description    : Main program.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
int main(void)
{
  
#ifdef DEBUG
  debug();
#endif

  /* System Clocks Configuration **********************************************/
  RCC_Configuration();   

  /* NVIC Configuration *******************************************************/
  NVIC_Configuration();

  /* SysTick end of count event each 1ms with input clock equal to 9MHz (HCLK/8, default) */
  SysTick_SetReload(9000);

  /* Enable SysTick interrupt */
  SysTick_ITConfig(ENABLE);

  /* Initial LCD 16x2 */
  lcd_init();											// Initial LCD
  
  // Loop Print Message to LCD16 x 2 //
  while(1)	  											// Loop Continue
  {
    goto_cursor(0x00);									// Set Cursor Line-1
    lcd_print("ET-STM32F103RBT6");						// Display LCD Line-1    
    goto_cursor(0x40);									// Set Cursor = Line-2
    lcd_print("ST-ARM Cortex-M3");						// Display LCD Line-2
    DelaymS(2500);										// Display Delay

	goto_cursor(0x00);									// Set Cursor Line-1
    lcd_print("32 Bit Processor");						// Display LCD Line-1    
    goto_cursor(0x40);									// Set Cursor = Line-2
    lcd_print("BY...ETT CO.,LTD");						// Display LCD Line-2
    DelaymS(2500);										// Display Delay  
  }  
}

/*******************************************************************************
* Function Name  : RCC_Configuration
* Description    : Configures the different system clocks.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void RCC_Configuration(void)
{   
  /* RCC system reset(for debug purpose) */
  RCC_DeInit();

  /* Enable HSE */
  RCC_HSEConfig(RCC_HSE_ON);

  /* Wait till HSE is ready */
  HSEStartUpStatus = RCC_WaitForHSEStartUp();

  if(HSEStartUpStatus == SUCCESS)
  {
    /* Enable Prefetch Buffer */
    FLASH_PrefetchBufferCmd(FLASH_PrefetchBuffer_Enable);

    /* Flash 2 wait state */
    FLASH_SetLatency(FLASH_Latency_2);
 	
    /* HCLK = SYSCLK */
    RCC_HCLKConfig(RCC_SYSCLK_Div1); 
  
    /* PCLK2 = HCLK */
    RCC_PCLK2Config(RCC_HCLK_Div1); 

    /* PCLK1 = HCLK/2 */
    RCC_PCLK1Config(RCC_HCLK_Div2);

    /* PLLCLK = 8MHz * 9 = 72 MHz */
    RCC_PLLConfig(RCC_PLLSource_HSE_Div1, RCC_PLLMul_9);

    /* Enable PLL */ 
    RCC_PLLCmd(ENABLE);

    /* Wait till PLL is ready */
    while(RCC_GetFlagStatus(RCC_FLAG_PLLRDY) == RESET)
    {
    }

    /* Select PLL as system clock source */
    RCC_SYSCLKConfig(RCC_SYSCLKSource_PLLCLK);

    /* Wait till PLL is used as system clock source */
    while(RCC_GetSYSCLKSource() != 0x08)
    {
    }
  }
}

/*******************************************************************************
* Function Name  : NVIC_Configuration
* Description    : Configures Vector Table base location.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void NVIC_Configuration(void)
{
#ifdef  VECT_TAB_RAM  
  /* Set the Vector Table base location at 0x20000000 */ 
  NVIC_SetVectorTable(NVIC_VectTab_RAM, 0x0); 
#else  /* VECT_TAB_FLASH  */
  /* Set the Vector Table base location at 0x08000000 */ 
  NVIC_SetVectorTable(NVIC_VectTab_FLASH, 0x0);   
#endif
}

/****************************/
/* Strobe 4-Bit Data to LCD */
/****************************/
void lcd_out_data4(unsigned char val)
{   
  if((val&0x01)==0x01)	// Bit[0] 
  {
    LCD_D4_HI();  
  }  
  else
  {
    LCD_D4_LO();
  }
  
  if((val&0x02)==0x02)  // Bit[1] 
  {
    LCD_D5_HI();  
  }  
  else
  {
    LCD_D5_LO();
  }
  
  if((val&0x04)==0x04)  // Bit[2] 
  {
    LCD_D6_HI();  
  }  
  else
  {
    LCD_D6_LO();
  } 
  
  if((val&0x08)==0x08)  // Bit[3]
  {
    LCD_D7_HI();  
  }  
  else
  {
    LCD_D7_LO();
  } 
  
}

/****************************/
/* Write Data 1 Byte to LCD */
/****************************/
void lcd_write_byte(unsigned char val)
{  
  lcd_out_data4((val>>4)&0x0F);							// Strobe 4-Bit High-Nibble to LCD
  enable_lcd();											// Enable Pulse
  
  lcd_out_data4(val&0x0F);				  				// Strobe 4-Bit Low-Nibble to LCD
  enable_lcd();											// Enable Pulse  

  while(busy_lcd());      								// Wait LCD Execute Complete  
}

/****************************/
/* Write Instruction to LCD */
/****************************/
void lcd_write_control(unsigned char val)
{ 
  LCD_RS_LO();											// RS = 0 = Instruction Select
  lcd_write_byte(val);									// Strobe Command Byte	    
}

/****************************/
/* Write Data(ASCII) to LCD */
/****************************/
void lcd_write_ascii(unsigned char c)
{  
  LCD_RS_HI();											// RS = 1 = Data Select
  lcd_write_byte(c);		   							// Strobe 1 Byte to LCD    
}

/*******************************/
/* Initial 4-Bit LCD Interface */
/*******************************/
void lcd_init(void)
{
  /* Configure IO connected to LCD16X2 */
  RCC_APB2PeriphClockCmd(RCC_APB2Periph_GPIO_EN, ENABLE);
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_Out_PP;
  GPIO_InitStructure.GPIO_Pin = LCD_EN_PIN;
  GPIO_Init(LCD_EN_PORT, &GPIO_InitStructure);

  RCC_APB2PeriphClockCmd(RCC_APB2Periph_GPIO_RW, ENABLE);
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_Out_PP;
  GPIO_InitStructure.GPIO_Pin = LCD_RW_PIN;
  GPIO_Init(LCD_RW_PORT, &GPIO_InitStructure);

  RCC_APB2PeriphClockCmd(RCC_APB2Periph_GPIO_RS, ENABLE);
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_Out_PP;
  GPIO_InitStructure.GPIO_Pin = LCD_RS_PIN;
  GPIO_Init(LCD_RS_PORT, &GPIO_InitStructure);

  RCC_APB2PeriphClockCmd(RCC_APB2Periph_GPIO_DATA, ENABLE);
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_Out_PP;
  GPIO_InitStructure.GPIO_Pin = LCD_D4_PIN |
                                LCD_D5_PIN |
								LCD_D6_PIN |
                                LCD_D7_PIN;
  GPIO_Init(LCD_DATA_PORT, &GPIO_InitStructure);

  LCD_D4_HI();
  LCD_D5_HI();
  LCD_D6_LO();
  LCD_D7_LO();
  DelayuS(15000);										// Power-On Delay (15 mS)
  
  LCD_D4_HI();
  LCD_D5_HI();
  LCD_D6_LO();
  LCD_D7_LO();
  enable_lcd();											// Enable Pulse
  DelayuS(4100);										// Delay 4.1mS
  
  LCD_D4_HI();
  LCD_D5_HI();
  LCD_D6_LO();
  LCD_D7_LO();
  enable_lcd();											// Enable Pulse
  DelayuS(100);											// delay 100uS
   
  LCD_D4_HI();
  LCD_D5_HI();
  LCD_D6_LO();
  LCD_D7_LO();
  enable_lcd();											// Enable Pulse
  while(busy_lcd());      								// Wait LCD Execute Complete
   
  LCD_D4_LO();
  LCD_D5_HI();
  LCD_D6_LO();
  LCD_D7_LO();
  enable_lcd();											// Enable Pulse
  while(busy_lcd());      								// Wait LCD Execute Complete
       
  lcd_write_control(0x28);  							// Function Set (DL=0 4-Bit,N=1 2 Line,F=0 5X7)
  lcd_write_control(0x0C);  							// Display on/off Control (Entry Display,Cursor off,Cursor not Blink)
  lcd_write_control(0x06);  							// Entry Mode Set (I/D=1 Increment,S=0 Cursor Shift)
  lcd_write_control(0x01);  							// Clear Display  (Clear Display,Set DD RAM Address=0) 
  DelaymS(15);  										// Wait Command Ready  
}

/***************************/
/* Set LCD Position Cursor */
/***************************/
void goto_cursor(unsigned char i)
{
  i |= 0x80;											// Set DD-RAM Address Command
  lcd_write_control(i);  
}

/************************************/
/* Print Display Data(ASCII) to LCD */
/************************************/
void lcd_print(unsigned char* str)
{
  int i;

  for (i=0;i<16 && str[i]!=0;i++)  						// 16 Character Print
  {
    lcd_write_ascii(str[i]);							// Print Byte to LCD
  }
}

/******************/
/* Wait LCD Ready */
/******************/

char busy_lcd(void)
{  
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;		// Config D7 = Read
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_IN_FLOATING;
  GPIO_InitStructure.GPIO_Pin = LCD_D7_PIN;
  GPIO_Init(LCD_DATA_PORT, &GPIO_InitStructure);

  LCD_RS_LO();		 									// Instruction Select
  LCD_RW_HI(); 											// Read Direction
  LCD_EN_HI();											// Start Read Busy

  DelayuS(100);	  										// Delay Before Read
  if (GPIO_ReadInputDataBit(LCD_DATA_PORT, LCD_D7_PIN) == Bit_SET)
  {
    LCD_EN_LO();  										// Disable Read
  	LCD_RW_LO();										// Default = Write Direction
  	
	GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;  	// Config D7 = Write
    GPIO_InitStructure.GPIO_Mode = GPIO_Mode_Out_PP;
    GPIO_InitStructure.GPIO_Pin = LCD_D7_PIN;
    GPIO_Init(LCD_DATA_PORT, &GPIO_InitStructure);
    return 1;											// LCD Busy Status
  }
  else
  {
    LCD_EN_LO();  										// Disable Read
  	LCD_RW_LO();										// Default = Write Direction
  	
	GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;	// Config D7 = Write
    GPIO_InitStructure.GPIO_Mode = GPIO_Mode_Out_PP;
    GPIO_InitStructure.GPIO_Pin = LCD_D7_PIN;
    GPIO_Init(LCD_DATA_PORT, &GPIO_InitStructure);
    return 0;											// LCD Ready Status
  }  
}


/***********************/
/* Enable Pulse to LCD */
/***********************/
void enable_lcd(void)	 								// Enable Pulse
{  
  LCD_EN_HI();  										// Enable ON
  DelayuS(50);  
  LCD_EN_LO();  										// Enable OFF 
}

/*******************************************************************************
* Function Name  : Delay
* Description    : Inserts a delay time.
* Input          : nCount: specifies the delay time length.
* Output         : None
* Return         : None
*******************************************************************************/
void DelayuS(vu32 nCount)
{  
  while (nCount--);
}

/*******************************************************************************
* Function Name  : Delay
* Description    : Inserts a delay time.
* Input          : nTime: specifies the delay time length, in milliseconds.
* Output         : None
* Return         : None
*******************************************************************************/
void DelaymS(u32 nTime)
{
  /* Enable the SysTick Counter */
  SysTick_CounterCmd(SysTick_Counter_Enable);
  
  TimingDelay = nTime;

  while(TimingDelay != 0);

  /* Disable SysTick Counter */
  SysTick_CounterCmd(SysTick_Counter_Disable);
  /* Clear SysTick Counter */
  SysTick_CounterCmd(SysTick_Counter_Clear);
}

/*******************************************************************************
* Function Name  : TimingDelay_Decrement
* Description    : Decrements the TimingDelay variable.
* Input          : None
* Output         : TimingDelay
* Return         : None
*******************************************************************************/
void TimingDelay_Decrement(void)
{
  if (TimingDelay != 0x00)
  { 
    TimingDelay--;
  }
}

#ifdef  DEBUG
/*******************************************************************************
* Function Name  : assert_failed
* Description    : Reports the name of the source file and the source line number
*                  where the assert_param error has occurred.
* Input          : - file: pointer to the source file name
*                  - line: assert_param error line source number
* Output         : None
* Return         : None
*******************************************************************************/
void assert_failed(u8* file, u32 line)
{ 
  /* User can add his own implementation to report the file name and line number,
     ex: printf("Wrong parameters value: file %s on line %d\r\n", file, line) */

  /* Infinite loop */
  while (1)
  {
  }
}
#endif

/******************* (C) COPYRIGHT 2008 STMicroelectronics *****END OF FILE****/
